<?php

namespace Sendcrux\Models;

global $wpdb;

use Illuminate\Database\Eloquent\Model as Eloquent;
use Sendcrux\Library\Log as SendcruxLog;

class Job extends Eloquent
{
    /* run by console only, init SendcruxLog first */
    public static function run()
    {
        $jobs = self::where('status', '=', 'new')->get();
        foreach ($jobs as $job) {
            SendcruxLog::info("Pushing user #{$job->user_id} to remote server (job ID: {$job->id})");
            $job->sync();
        }
    }

    public function setRunning()
    {
        $this->status = 'running';
        $this->save();
    }

    /**
     * Actually execute the job, sync the corresponding user's details to Sendcrux Mail
     */
    public function sync()
    {
        $this->setRunning();
        $conn = Connection::first();
        switch ($this->operation) {
            case 'create':
            case 'update':
                try {
                    $conn->request('POST', "subscribers", $this->buildCreateUpdateParams($conn->getListUid()));
                    $this->log('finished', 'Successfully');
                } catch (\Exception $ex) {
                    $this->log('failed', $ex->getMessage());
                } finally {
                    $this->delete();
                }
                break;
            case 'delete':
                try {
                    $response = $conn->request('GET', "subscribers/email/{$this->getUser()['user_email']}");
                    $get_uid = $this->getSubscriberUid($response,$conn->getListUid());
                    $conn->request('DELETE', "subscribers/{$get_uid}");
                    $this->log('finished', 'Successfully');
                } catch (\Exception $ex) {
                    $this->log('failed', $ex->getMessage());
                } finally {
                    $this->delete();
                }
                break;
        }
    }

    public static function reset()
    {
        $users = get_users();
        foreach ($users as $user) {
            $job = new Job();
            $job->operation = 'update';
            $job->user_id = $user->data->ID;
            $job->status = 'new';
            $job->save();
        }
    }

    public function log($status, $description)
    {
        $log = new \Sendcrux\Models\Log();
        $user = $this->getUser();
        $log->user_id = $this->user_id;
        $log->user_email = $user['user_email'];
        $log->user_nicename = $user['user_nicename'];
        $log->operation = $this->operation;
        $log->status = $status;
        $log->description = $description;
        $log->save();
    }

    public function buildCreateUpdateParams($uid)
    {
        $conn = Connection::first();
        $mappings = json_decode($conn->settings, true)['mapping'];
        $body = [];
        $details = $this->getUser();

        foreach ($mappings as $mapping) {
            $body[$mapping['sendcrux-field']] = $details[$mapping['wp-field']];
        }

        $body['list_uid'] = $uid;

        return $body;
    }

    public function getSubscriberUid($response,$list_uid){
        
        foreach($response->subscribers as $item){
            if($item->list_uid == $list_uid){
                return $item->uid;
            }
        }
    }

    public function getUser()
    {
        $metaValues = array_map(function ($r) {
            return $r[0];
        }, get_user_meta($this->user_id));
        $u = get_user_by('ID', $this->user_id);
        $userValues = json_decode(json_encode($u->data), true);
        $details = array_merge($userValues, $metaValues);
        return $details;
    }

    public static function getItems($params = [])
    {
        if (empty($params)) {
            $items = self::offset(0);
        } else {
            $items = self::limit($params['per_page'])->offset($params['per_page'] * ($params['current_page'] - 1));
            if (array_key_exists('orderby', $_GET)) {
                $items = $items->orderBy($_GET['orderby'], $_GET['order']);
            }
        }
        $items = $items->get()->map(function ($r) {
            return $r->toCustomArray();
        })->toArray();

        return $items;
    }
    
    public function toCustomArray()
    {
        $r = $this->toArray();
        $user = $this->getUser();
        $r['user_email'] = $user['user_email'];
        $r['user_nicename'] = $user['user_nicename'];
        return $r;
    }
    
    public static function getLastUpdate()
    {
        $r = self::orderBy('created_at', 'desc')->first();
        if (empty($r)) {
            return null;
        } else {
            return $r->created_at->diffForHumans();
        }
    }
}
